package com.icetech.oss.impl;

import com.amazonaws.ClientConfiguration;
import com.amazonaws.HttpMethod;
import com.amazonaws.Protocol;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.client.builder.AwsClientBuilder;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.GeneratePresignedUrlRequest;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.PutObjectResult;
import com.amazonaws.services.s3.model.S3Object;
import com.icetech.oss.OSSProperties;
import lombok.extern.slf4j.Slf4j;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.time.Instant;
import java.util.Date;

@Slf4j
public class CtyunOssService extends BaseOssService<AmazonS3> {

    @Override
    public boolean isExistFile(String fileName, String bucketName) {
        try {
            InputStream oss2InputStream = getOSS2InputStream(fileName, bucketName);
            return oss2InputStream != null;
        } catch (Exception e) {
            log.error("OSS文件是否存在获取失败|{}|{}", bucketName, fileName, e);
        }
        return false;
    }

    @Override
    public String buildImageUrl(String bucketName, String fileName) {
        return getImageUrl(bucketName, fileName, 3600L);
    }

    @Override
    public String getInternalImageUrl(String name, String bucketName, Long expireSec) {
        return getImageUrl(bucketName, name, expireSec);
    }

    @Override
    public String getImageUrl(String bucketName, String name, Long expireSec) {
        try {
            if (expireSec == null || expireSec < 1) expireSec = ossProperties.getDefaultExpireSeconds();
            // 生成URL
            //获取预签名Url
            GeneratePresignedUrlRequest var = new
                    GeneratePresignedUrlRequest(bucketName, name);
            Date expiration = new Date();
            long expTimeMillis = Instant.now().toEpochMilli();
            expTimeMillis += 1000 * expireSec;
            // 设置有效期60分钟
            expiration.setTime(expTimeMillis);
            var.setExpiration(expiration);
            var.setMethod(HttpMethod.GET);
            URL url = publicOssClient.generatePresignedUrl(var);
            return url.toString().replaceAll("^http://", "https://");
        } catch (Exception e) {
            log.error("OSS文件生成访问地址失败|{}|{}|{}", bucketName, name, expireSec, e);
        }
        return null;
    }

    @Override
    public InputStream getOSS2InputStream(String name, String bucketName) {
        try {
            GetObjectRequest request = new GetObjectRequest(bucketName, name);
            S3Object result = publicOssClient.getObject(request);
            return result.getObjectContent();
        } catch (Exception e) {
            log.error("OSS文件获取下载流失败|{}|{}", bucketName, name, e);
        }
        return null;
    }

    @Override
    public boolean getOSS2File(String fileName, String bucketName, File stored) {
        try {
            if (stored.exists() && !stored.delete()) {
                log.warn("OSS文件下载存储失败, 本地文件已存在且删除失败|{}|{}|{}", bucketName, fileName, stored.getPath());
                return false;
            }
            GetObjectRequest request = new GetObjectRequest(bucketName, fileName);
            S3Object result = publicOssClient.getObject(request);
            try (InputStream in = result.getObjectContent()) {
                try (FileOutputStream outputStream = new
                        FileOutputStream(stored)) {
                    byte[] readBuf = new byte[1024 * 1024];
                    int readLen = 0;
                    while ((readLen = in.read(readBuf)) > 0) {
                        outputStream.write(readBuf, 0, readLen);
                    }
                }
            } catch (IOException e){
                log.warn("OSS文件下载存储出错|{}|{}|{}", bucketName, fileName, stored.getPath(), e);
            }
            log.info("OSS文件下载存储成功|{}|{}|{}", bucketName, fileName, stored.getPath());
            return true;
        } catch (Exception e) {
            log.error("OSS文件下载存储失败|{}|{}|{}", bucketName, fileName, stored.getPath(), e);
        }
        return false;
    }

    @Override
    public boolean uploadFile(File file, String fileName, String bucketName) {
        try {
            PutObjectRequest request = new PutObjectRequest(bucketName, fileName, file);
            PutObjectResult putObjectResult = publicOssClient.putObject(request);
            log.info("OSS文件上传成功|{}|{}|{}|{}|{}", file.getPath(), bucketName, fileName, putObjectResult.getETag(), putObjectResult.getVersionId());
            return true;
        } catch (Exception e) {
            log.error("OSS文件上传失败|{}|{}|{}", file.getPath(), bucketName, fileName, e);
        }
        return false;
    }

    @Override
    public boolean uploadFile(InputStream inputStream, String fileName, String bucketName) {
        try {
            PutObjectRequest request = new PutObjectRequest(bucketName, fileName, inputStream, null);
            PutObjectResult putObjectResult = publicOssClient.putObject(request);
            log.info("OSS输入流上传成功|{}|{}|{}|{}", bucketName, fileName, putObjectResult.getETag(), putObjectResult.getVersionId());
            return true;
        } catch (Exception e) {
            log.error("OSS输入流上传失败|{}|{}", bucketName, fileName, e);
        }
        return false;
    }

    public CtyunOssService(OSSProperties ossProperties) {
        super(ossProperties, properties -> {
            AWSCredentials credentials = new BasicAWSCredentials(ossProperties.getAccessKeyId(),
                    ossProperties.getAccessKeySecret());
            ClientConfiguration awsClientConfig = new ClientConfiguration();
            awsClientConfig.setSignerOverride("AWSS3V4SignerType");
            awsClientConfig.setProtocol(Protocol.HTTP);
            return AmazonS3ClientBuilder.standard()
                    .withCredentials(new
                            AWSStaticCredentialsProvider(credentials))
                    .withClientConfiguration(awsClientConfig)
                    .withEndpointConfiguration(new
                            AwsClientBuilder.EndpointConfiguration(ossProperties.getEndpoint(), ""))
                    .disableChunkedEncoding()
                    .enablePathStyleAccess()
                    .build();
        },properties -> {
            AWSCredentials credentials = new BasicAWSCredentials(ossProperties.getAccessKeyId(),
                    ossProperties.getAccessKeySecret());
            ClientConfiguration awsClientConfig = new ClientConfiguration();
            awsClientConfig.setSignerOverride("AWSS3V4SignerType");
            awsClientConfig.setProtocol(Protocol.HTTP);
            return AmazonS3ClientBuilder.standard()
                    .withCredentials(new
                            AWSStaticCredentialsProvider(credentials))
                    .withClientConfiguration(awsClientConfig)
                    .withEndpointConfiguration(new
                            AwsClientBuilder.EndpointConfiguration(ossProperties.getEndpoint(), ""))
                    .disableChunkedEncoding()
                    .enablePathStyleAccess()
                    .build();
        });
        log.info("ctyun OSS客户端初始化成功|{}|{}|{}", ossProperties.getEndpoint(), ossProperties.getInternalEndpoint(), ossProperties.getBucketName());
    }
}
