package com.icetech.oss.impl;

import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.util.Date;

import com.amazonaws.ClientConfiguration;
import com.amazonaws.Protocol;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.S3ClientOptions;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectResult;
import com.amazonaws.services.s3.model.S3Object;
import com.icetech.oss.OSSProperties;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class AmazonS3OssService extends BaseOssService<AmazonS3Client> {
    @Override
    public boolean isExistFile(String fileName, String bucketName) {
        try {
            return internalOssClient.doesObjectExist(bucketName, fileName);
        } catch (Exception e) {
            log.error("[oss异常]e", e);
        }
        return false;
    }

    @Override
    public String buildImageUrl(String bucketName, String fileName) {
        return null;
    }

    @Override
    public String getImageUrl(String bucketName, String name, Long expireSec) {
        try {
            // 设置URL过期时间为1小时
            Date expiration = new Date(System.currentTimeMillis() + expireSec * 1000);
            // 生成URL
            URL url = publicOssClient.generatePresignedUrl(bucketName, name, expiration);
            return url.toString().replace("http://", "https://");
        } catch (Exception e) {
            log.error("[oss异常]e", e);
        }
        return null;
    }

    @Override
    public String getInternalImageUrl(String name, String bucketName, Long expireSec) {
        try {
            // 设置URL过期时间为1小时
            Date expiration = new Date(System.currentTimeMillis() + expireSec * 1000);
            // 生成URL
            URL url = internalOssClient.generatePresignedUrl(bucketName, name, expiration);
            return url.toString().replace("http://", "https://");
        } catch (Exception e) {
            log.error("[oss异常]e", e);
        }
        return null;
    }

    @Override
    public InputStream getOSS2InputStream(String name, String bucketName) {
        try {
            S3Object ossObj = internalOssClient.getObject(ossProperties.getBucketName(), name);
            return ossObj == null ? null : ossObj.getObjectContent();
        } catch (Exception e) {
            log.error("[oss异常]e", e);
        }
        return null;
    }

    @Override
    public boolean getOSS2File(String fileName, String bucketName, File file) {
        try {
            internalOssClient.getObject(new GetObjectRequest(bucketName, fileName), file);
            return true;
        } catch (Exception e) {
            log.error("[oss异常]e", e);
        }
        return false;
    }

    @Override
    public boolean uploadFile(File file, String fileName, String bucketName) {
        try {
            PutObjectResult result = internalOssClient.putObject(bucketName, fileName, file);
            log.info("OSS文件上传成功|{}|{}|{}|{}|{}", file.getPath(), bucketName, fileName, result.getETag(), result.getVersionId());
            return true;
        } catch (Exception e) {
            log.error("OSS文件上传失败|{}|{}|{}", file.getPath(), bucketName, fileName, e);
        }
        return false;
    }

    @Override
    public boolean uploadFile(InputStream inputStream, String fileName, String bucketName) {
        try {
            PutObjectResult result = internalOssClient.putObject(bucketName, fileName, inputStream, buildMateDate(inputStream.available()));
            log.info("OSS输入流上传成功|{}|{}|{}|{}", bucketName, fileName, result.getETag(), result.getVersionId());
            return true;
        } catch (Exception e) {
            log.error("OSS输入流上传失败|{}|{}", bucketName, fileName, e);
        }
        return false;
    }

    private ObjectMetadata buildMateDate(long size) {
        ObjectMetadata metadata = new ObjectMetadata();
        metadata.setContentLength(size);
        metadata.setContentType("image/jpeg");
        return metadata;
    }

    public AmazonS3OssService(OSSProperties ossProperties) {
        super(ossProperties, properties -> {
            BasicAWSCredentials basicAwsCred = new BasicAWSCredentials(ossProperties.getAccessKeyId(), ossProperties.getAccessKeySecret()); //请替换成您的AccessKey和SecretKey
            ClientConfiguration clientConfiguration = new ClientConfiguration();
            clientConfiguration.setProtocol(Protocol.valueOf(ossProperties.getProtocol()));
            AmazonS3Client client = new AmazonS3Client(basicAwsCred, clientConfiguration);
            S3ClientOptions options = S3ClientOptions.builder().setPathStyleAccess(true).setPayloadSigningEnabled(true).disableChunkedEncoding().build();
            client.setS3ClientOptions(options);
            client.setEndpoint(ossProperties.getEndpoint());
            return client;
        }, properties -> {
            BasicAWSCredentials basicAwsCred = new BasicAWSCredentials(ossProperties.getAccessKeyId(), ossProperties.getAccessKeySecret()); //请替换成您的AccessKey和SecretKey
            ClientConfiguration clientConfiguration = new ClientConfiguration();
            clientConfiguration.setProtocol(Protocol.valueOf(ossProperties.getInternalProtocol()));
            AmazonS3Client client = new AmazonS3Client(basicAwsCred, clientConfiguration);
            S3ClientOptions options = S3ClientOptions.builder().setPathStyleAccess(true).setPayloadSigningEnabled(true).disableChunkedEncoding().build();
            client.setS3ClientOptions(options);
            client.setEndpoint(ossProperties.getInternalEndpoint());
            return client;
        });
    }
}
