package com.icetech.db.mybatis.base.mapper;

import java.util.List;
import java.util.stream.Collectors;

import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.core.toolkit.support.SFunction;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

/**
 * Description 公共mapper
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @param <T> the type parameter
 * @author wgy
 * @version 1.0.5
 * @since Date 2021/4/29 9:33 AM
 */
public interface SuperMapper<T> extends BaseMapper<T> {
    String LIMIT_ONE = "limit 1";

    /**
     * <p>根据实体类查询单条记录</p>
     * <p>SQL层强制返回单条记录, 仅限于MySQL</p>
     *
     * @param entity 实体类, 非空字段为查询条件
     * @return 查询结果
     */
    default T selectLimitOne(T entity) {
        return selectLimitOne(Wrappers.query(entity));
    }

    /**
     * <p>根据查询条件查询单条记录</p>
     * <p>SQL层强制返回单条记录, 仅限于MySQL</p>
     *
     * @param wrapper 查询条件
     * @return 查询结果
     */
    default T selectLimitOne(QueryWrapper<T> wrapper) {
        return selectOne(wrapper.last(LIMIT_ONE));
    }

    /**
     * <p>根据查询条件查询单条记录</p>
     * <p>SQL层强制返回单条记录, 仅限于MySQL</p>
     *
     * @param wrapper 查询条件
     * @return 查询结果
     */
    default T selectLimitOne(LambdaQueryWrapper<T> wrapper) {
        return selectOne(wrapper.last(LIMIT_ONE));
    }

    /**
     * <p>根据实体类查询多条记录</p>
     *
     * @param entity 实体类, 非空字段为查询条件
     * @return 查询结果
     */
    default List<T> selectList(T entity) {
        return selectList(Wrappers.query(entity));
    }

    /**
     * <p>根据实体类分页查询多条记录</p>
     *
     * @param entity  实体类, 非空字段为查询条件
     * @param current 当前页数
     * @param size    每页大小
     * @return 分页查询结果
     */
    default Page<T> selectPage(T entity, long current, long size) {
        return selectPage(Wrappers.query(entity), current, size);
    }

    /**
     * <p>根据查询条件分页查询多条记录</p>
     *
     * @param wrapper 查询条件
     * @param current 当前页数
     * @param size    每页大小
     * @return 分页查询结果
     */
    default Page<T> selectPage(Wrapper<T> wrapper, long current, long size) {
        return selectPage(new Page<>(current, size), wrapper);
    }

    default <C> C selectLimitOneColumn(SFunction<T, C> column, T entity) {
        return selectLimitOneColumn(column, Wrappers.lambdaQuery(entity));
    }

    default <C> C selectLimitOneColumn(SFunction<T, C> column, LambdaQueryWrapper<T> wrapper) {
        T result = selectLimitOne(wrapper.select(column));
        return result == null ? null : column.apply(result);
    }

    default <C> C selectOneColumn(SFunction<T, C> column, T entity) {
        return selectOneColumn(column, Wrappers.lambdaQuery(entity));
    }

    default <C> C selectOneColumn(SFunction<T, C> column, LambdaQueryWrapper<T> wrapper) {
        T result = selectOne(wrapper.select(column));
        return result == null ? null : column.apply(result);
    }

    default <C> List<C> selectColumnList(SFunction<T, C> column, T entity) {
        return selectColumnList(column, Wrappers.lambdaQuery(entity));
    }

    default <C> List<C> selectColumnList(SFunction<T, C> column, LambdaQueryWrapper<T> wrapper) {
        List<T> results = selectList(wrapper.select(column));
        return results.stream().map(column).collect(Collectors.toList());
    }
}
