package com.icetech.db.config;

import com.baomidou.mybatisplus.annotation.DbType;
import com.baomidou.mybatisplus.extension.plugins.MybatisPlusInterceptor;
import com.baomidou.mybatisplus.extension.plugins.handler.TenantLineHandler;
import com.baomidou.mybatisplus.extension.plugins.inner.PaginationInnerInterceptor;
import com.github.pagehelper.PageInterceptor;
import com.icetech.common.properties.TenantProperties;
import com.icetech.db.health.DbHealthIndicator;
import com.icetech.db.interceptor.CustomTenantInterceptor;
import com.icetech.db.mybatis.DateMetaObjectHandler;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * Description DbAutoConfiguration
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wgy
 * @version 1.0 @Date 2021/4/29 5:44 PM
 */
@Slf4j
@Configuration
public class DbAutoConfiguration {

    @Autowired
    private TenantProperties tenantProperties;

    @Autowired
    private TenantLineHandler tenantLineHandler;

    /**
     * mybatis 分页插件
     * @return MybatisPlusInterceptor
     */
    @Bean
    @ConditionalOnMissingBean(MybatisPlusInterceptor.class)
    public MybatisPlusInterceptor mybatisPlusInterceptor() {
        MybatisPlusInterceptor interceptor = new MybatisPlusInterceptor();
        boolean enableTenant = tenantProperties.getEnable();
        //是否开启多租户隔离
        if (enableTenant) {
            CustomTenantInterceptor tenantInterceptor = new CustomTenantInterceptor(
                    tenantLineHandler, tenantProperties.getIgnoreSqlList());
            interceptor.addInnerInterceptor(tenantInterceptor);
        }
        interceptor.addInnerInterceptor(new PaginationInnerInterceptor(DbType.MYSQL));
        return interceptor;
    }

    /**
     * pagehelper 分页插件
     * @return PageInterceptor
     */
    @Bean
    @ConditionalOnMissingBean(PageInterceptor.class)
    public PageInterceptor pageInterceptor() {
        return new PageInterceptor();
    }

    /**
     * Description:  db 健康检测
     * 
     * @return db.health.DbHealthIndicator       
     */
    @Bean
    @ConditionalOnMissingBean(DbHealthIndicator.class)
    public DbHealthIndicator dbHealthIndicator() {
        DbHealthIndicator dbHealthIndicator = new DbHealthIndicator();
        log.info("DbHealthIndicator [{}]", dbHealthIndicator);
        return dbHealthIndicator;
    }

    /**
     * 自动填充模型数据
     *
     */
    @Bean
    @ConditionalOnMissingBean(DateMetaObjectHandler.class)
    public DateMetaObjectHandler dateMetaObjectHandler() {
        DateMetaObjectHandler metaObjectHandler = new DateMetaObjectHandler();
        log.info("DateMetaObjectHandler [{}]", metaObjectHandler);
        return metaObjectHandler;
    }

}
