package com.icetech.common.utils;

import cn.hutool.core.util.ReflectUtil;
import sun.reflect.generics.reflectiveObjects.ParameterizedTypeImpl;

import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class ReflectUtils extends ReflectUtil {
    /**
     * <p>获取接口定义泛型</p>
     *
     * @param interfaceClass 泛型接口类Class(<b>必须为泛型接口</b>)
     * @param implement      泛型接口实现类对象(<b>必须指明所实现接口的泛型类型</b>)
     * @param <T>            接口类型
     * @return 接口类泛型列表
     * @throws IllegalArgumentException 找不到泛型类型(1.接口非泛型接口; 2.实现类未指明泛型类型)
     */
    public static <T> Type[] getInterfaceGenericTypes(Class<T> interfaceClass, T implement) {
        return getInterfaceGenericTypes(implement.getClass(), interfaceClass);
    }

    private static final Map<String, Type[]> INTERFACE_GENERIC_TYPES_CACHE = new ConcurrentHashMap<>(64);
    public static Type[] getInterfaceGenericTypes(Class<?> implementClass, Class<?> interfaceClass) {
        String cacheKey = interfaceClass.getTypeName() + ":" + implementClass.getTypeName();
        return INTERFACE_GENERIC_TYPES_CACHE.computeIfAbsent(cacheKey, key -> {
            Type[] typeArguments = getActualTypeArguments(implementClass.getGenericInterfaces(), interfaceClass);
            if (typeArguments == null) {
                List<Class<?>> superClasses = getAllSuperClasses(implementClass);
                Type[] types = superClasses.stream().flatMap(superClass -> Stream.of(superClass.getGenericInterfaces())).toArray(Type[]::new);
                typeArguments = getActualTypeArguments(types, interfaceClass);
            }
            if (typeArguments == null) {
                throw new IllegalArgumentException("can not find " + implementClass.getName() + " Implements generic type");
            }
            return typeArguments;
        });
    }

    public static Type[] getActualTypeArguments(Type[] genericInterfaces, Class<?> interfaceClass) {
        Type[] typeArguments = null;
        for (Type type : genericInterfaces) {
            if (type instanceof ParameterizedType) {
                ParameterizedType parameterizedType = (ParameterizedType) type;
                Type rawType = parameterizedType.getRawType();
                if (rawType instanceof Class && interfaceClass.isAssignableFrom((Class<?>) rawType)) {
                    typeArguments = parameterizedType.getActualTypeArguments();
                }
            }
        }
        return typeArguments;
    }

    /**
     * <p>获取接口定义泛型并包装到另一个泛型中</p>
     *
     * @param interfaceClass      泛型接口类Class(<b>必须为泛型接口</b>)
     * @param implement           泛型接口实现类对象(<b>必须指明所实现接口的泛型类型</b>)
     * @param genericWrapperClass 包装类Class(<b>包装类也需为泛型, 且泛型数量及顺序与接口声明一致</b>)
     * @param <T>                 接口类型
     * @return 接口类泛型列表
     * @throws IllegalArgumentException 找不到泛型类型(1.接口非泛型接口; 2.实现类未指明泛型类型)
     */
    public static <T> Type getInterfaceGenericTypeWrapper(Class<T> interfaceClass, T implement, Class<?> genericWrapperClass) {
        Type[] types = getInterfaceGenericTypes(interfaceClass, implement);
        return getWrapperType(genericWrapperClass, types);
    }

    /**
     * <p>获取接口定义的首个泛型并包装到另一个泛型中</p>
     *
     * @param interfaceClass      泛型接口类Class(<b>必须为泛型接口</b>)
     * @param implement           泛型接口实现类对象(<b>必须指明所实现接口的泛型类型</b>)
     * @param genericWrapperClass 包装类Class(<b>包装类也需为泛型</b>)
     * @param <T>                 接口类型
     * @return 接口类泛型列表
     * @throws IllegalArgumentException 找不到泛型类型(1.接口非泛型接口; 2.实现类未指明泛型类型)
     */
    public static <T> Type getInterfaceFirstGenericTypeWrapper(Class<T> interfaceClass, T implement, Class<?> genericWrapperClass) {
        Type[] types = getInterfaceGenericTypes(interfaceClass, implement);
        return getWrapperType(genericWrapperClass, types[0]);
    }

    /**
     * <p>获取接口定义的泛型并包装到另一个泛型中</p>
     *
     * @param interfaceClass      泛型接口类Class(<b>必须为泛型接口</b>)
     * @param implement           泛型接口实现类对象(<b>必须指明所实现接口的泛型类型</b>)
     * @param genericWrapperClass 包装类Class(<b>包装类也需为泛型</b>)
     * @param index               接口泛型索引
     * @param <T>                 接口类型
     * @return 接口类泛型列表
     * @throws IllegalArgumentException 找不到泛型类型(1.接口非泛型接口; 2.实现类未指明泛型类型)
     */
    public static <T> Type getInterfaceFirstGenericTypeWrapper(Class<T> interfaceClass, T implement, Class<?> genericWrapperClass, int index) {
        Type[] types = getInterfaceGenericTypes(interfaceClass, implement);
        return getWrapperType(genericWrapperClass, types[index]);
    }

    private static final Map<String, Type> WRAPPER_TYPE_CACHE = new ConcurrentHashMap<>(256);

    /**
     * 获取包装泛型类型
     *
     * @param outClass   外层类型
     * @param innerClass 内层类型
     * @return
     */
    public static Type getWrapperType(Class<?> outClass, Class<?>... innerClass) {
        return getWrapperType(true, outClass, innerClass);
    }

    public static Type getWrapperType(boolean cache, Class<?> outClass, Class<?>... innerClass) {
        return getWrapperType(cache, outClass, (Type[]) innerClass);
    }

    /**
     * 获取包装泛型类型
     *
     * @param outClass  外层类型
     * @param innerType 内层类型
     * @return
     */
    public static Type getWrapperType(Class<?> outClass, Type... innerType) {
        return getWrapperType(true, outClass, innerType);
    }

    public static Type getWrapperType(boolean cache, Class<?> outClass, Type... innerType) {
        if (!cache) return ParameterizedTypeImpl.make(outClass, innerType, null);
        String cacheKey = outClass.getTypeName() + "&" + Stream.of(innerType).map(Type::getTypeName).collect(Collectors.joining(","));
        return WRAPPER_TYPE_CACHE.computeIfAbsent(cacheKey, key -> ParameterizedTypeImpl.make(outClass, innerType, null));
    }

    /**
     * 获取循环嵌套包装泛型类型
     *
     * @param outClass   外层类型
     * @param innerClass 内层类型
     * @return
     */
    public static Type getLoopWrapperType(Class<?> outClass, Class<?>... innerClass) {
        if (innerClass == null) {
            throw new IllegalArgumentException("clazz must be not null");
        }

        if (innerClass.length == 1) {
            return getWrapperType(outClass, innerClass[0]);
        }

        Type type = getWrapperType(innerClass[innerClass.length - 2], innerClass[innerClass.length - 1]);
        for (int i = innerClass.length - 3; i >= 0; i--) {
            type = getWrapperType(innerClass[i], type);
        }

        return getWrapperType(outClass, type);
    }

    public static List<Class<?>> getAllSuperClasses(Class<?> clazz) {
        List<Class<?>> list = new LinkedList<>();
        Class<?> superClass = clazz;
        while ((superClass = superClass.getSuperclass()) != null) {
            list.add(superClass);
        }
        return list;
    }

    private static final String SETTER_PREFIX = "set";

    private static final String GETTER_PREFIX = "get";

    /**
     * 调用Getter方法.
     * 支持多级，如：对象名.对象名.方法
     */
    @SuppressWarnings("unchecked")
    public static <E> E invokeGetter(Object obj, String propertyName) {
        Object object = obj;
        for (String name : StringUtils.split(propertyName, ".")) {
            String getterMethodName = GETTER_PREFIX + StringUtils.capitalize(name);
            object = invoke(object, getterMethodName);
        }
        return (E) object;
    }

    /**
     * 调用Setter方法, 仅匹配方法名。
     * 支持多级，如：对象名.对象名.方法
     */
    public static <E> void invokeSetter(Object obj, String propertyName, E value) {
        Object object = obj;
        String[] names = StringUtils.split(propertyName, ".");
        for (int i = 0; i < names.length; i++) {
            if (i < names.length - 1) {
                String getterMethodName = GETTER_PREFIX + StringUtils.capitalize(names[i]);
                object = invoke(object, getterMethodName);
            } else {
                String setterMethodName = SETTER_PREFIX + StringUtils.capitalize(names[i]);
                Method method = getMethodByName(object.getClass(), setterMethodName);
                invoke(object, method, value);
            }
        }
    }
}
