package com.icetech.common.domain;

import com.icetech.common.domain.response.R;
import com.icetech.common.utils.LockPoolUtils;
import com.icetech.common.utils.NumberUtils;
import com.icetech.common.utils.UUIDTools;

import java.math.BigInteger;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 *  redis 消息广播订阅
 *
 * @author fangct
 */
public interface AsyncNotifyInterface {
    /**
     * topic前缀
     */
    String TOPIC_PREFIX = "asyncNotify";
    /**
     * redis key前缀
     */
    String KEY_PREFIX = "notify:";

    String key = NumberUtils.radix62(new BigInteger(UUIDTools.getUuid(), 16));
    /**
     * 订阅topic(routing key)
     */
    String topic = TOPIC_PREFIX + "." + key;

    /**
     * 获取消息key
     * @return topic
     */
    static String getTopic() {
        return topic;
    }
    static String getKey() {
        return key;
    }
    /**
     * 获取消息key
     * @param key key
     * @return redis key
     */
    static String getMessageKey(String key) {
        return KEY_PREFIX + TOPIC_PREFIX + ":" + key;
    }

    /**
     * 接收的消息
     * @param msg 消息
     */
    void receiveMessage(String msg);

    /**
     * wait方法
     * @param lock
     * @param waitTime
     * @param supplier
     * @param <R>
     * @return
     */
    static <R> R wait(String lock, Long waitTime, Supplier<R> supplier) {
        try {
            LockPoolUtils.add(lock);
            synchronized (lock) {
                try {
                    lock.wait(waitTime == null ? 4000 : waitTime);
                } catch (InterruptedException e) {
                    return null;
                }
            }
            return supplier.get();
        } finally {
            LockPoolUtils.remove(lock);
        }
    }
    /**
     * notify方法
     * @param key
     * @param supplier
     * @return
     */
    static void notify(String key, String topic, Function<String, R> supplier) {
        if (topic == null) {
            return;
        }
        String lock = LockPoolUtils.getLock(key);
        if (lock != null) {
            synchronized (lock) {
                lock.notifyAll();
            }
        } else {
            supplier.apply(topic);
        }
    }
}
