package com.icetech.common.utils;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHeaders;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.lang.reflect.Array;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Slf4j
public class HttpUtils {
    private static final RequestConfig DEFAULT_REQUEST_CONFIG = RequestConfig.custom().setConnectTimeout(5000).setSocketTimeout(30000).build();

    public static String get(String url, String... param) throws IOException {
        if (param != null && param.length > 0) {
            url += "?" + buildUrlParam(param);
        }
        return request(new HttpGet(url));
    }

    public static String get(String url, Map<String, Object> params) throws IOException {
        if (params != null && !params.isEmpty()) {
            url += "?" + buildUrlParam(params);
        }
        return get(url);
    }

    public static String postForm(String url, String... param) throws IOException {
        if (param != null && param.length > 0) {
            url += "?" + buildUrlParam(param);
        }
        return get(url);
    }

    public static String post(String url, Map<String, Object> params) throws IOException {
        if (params == null || params.isEmpty()) {
            return postForm(url);
        }

        return post(url, params.entrySet().stream().flatMap(param -> {
            if (param.getValue() == null) {
                return Stream.of(new BasicNameValuePair(param.getKey(), ""));
            }

            if (param.getValue().getClass().isArray()) {
                int length = Array.getLength(param.getValue());
                List<BasicNameValuePair> pairs = new ArrayList<>(length);
                for (int i = 0; i < length; i++) {
                    Object value = Array.get(param.getValue(), i);
                    pairs.add(new BasicNameValuePair(param.getKey(), value == null ? "" : value.toString()));
                }
                return pairs.stream();
            }

            return Stream.of(new BasicNameValuePair(param.getKey(), param.getValue().toString()));
        }).collect(Collectors.toList()));
    }

    public static String post(String url, List<NameValuePair> params) throws IOException {
        HttpPost request = new HttpPost(url);
        if (params != null && !params.isEmpty()) {
            UrlEncodedFormEntity requestEntity = new UrlEncodedFormEntity(params, StandardCharsets.UTF_8);
            request.setEntity(requestEntity);
            request.setHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded;charset=" + StandardCharsets.UTF_8);
        }

        return request(request);
    }

    private static String request(HttpUriRequest request) throws IOException {
        try (CloseableHttpClient httpClient = HttpClientBuilder.create().setDefaultRequestConfig(DEFAULT_REQUEST_CONFIG).build()) {
            try (CloseableHttpResponse response = httpClient.execute(request)) {
                HttpEntity httpEntity = response.getEntity();
                return EntityUtils.toString(httpEntity, getResponseCharset(httpEntity));
            }
        }
    }

    private static String buildUrlParam(Map<String, Object> params) {
        return params.entrySet().stream().map(param -> {
            String content = param.getKey() + "=";
            if (param.getValue() != null) {
                if (param.getValue().getClass().isArray()) {
                    int length = Array.getLength(param.getValue());
                    StringJoiner joiner = new StringJoiner("&" + content);
                    for (int i = 0; i < length; i++) {
                        Object value = Array.get(param.getValue(), i);
                        joiner.add(value == null ? "" : value.toString());
                    }
                    content += joiner.toString();
                } else {
                    content += param.getValue();
                }
            }
            return content;
        }).collect(Collectors.joining("&"));
    }

    private static String buildUrlParam(String[] params) {
        StringJoiner joiner = new StringJoiner("&");
        for (int i = 0; i < params.length; i++) {
            String name = params[i];
            String value = ++i < params.length ? params[i] : "";
            joiner.add(name + "=" + value);
        }
        return joiner.toString();
    }

    private static final Pattern CONTENT_TYPE_CHARSET_PATTERN = Pattern.compile("charset=([A-Za-z0-9-]+)$");

    private static String getResponseCharset(HttpEntity entity) {
        if (entity.getContentEncoding() != null) {
            if (StringUtils.isNotBlank(entity.getContentEncoding().getValue())) {
                return entity.getContentEncoding().getValue();
            }
        }

        if (entity.getContentType() != null) {
            if (StringUtils.isNotBlank(entity.getContentType().getValue())) {
                String contentType = entity.getContentType().getValue();
                Matcher matcher = CONTENT_TYPE_CHARSET_PATTERN.matcher(contentType);
                if (matcher.find()) {
                    return matcher.group(1);
                }
            }
        }

        return StandardCharsets.UTF_8.name();
    }
}
