package com.icetech.city.road.domain.table;

import java.io.Serializable;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.Date;

import com.baomidou.mybatisplus.annotation.FieldStrategy;
import com.baomidou.mybatisplus.annotation.IdType;
import com.baomidou.mybatisplus.annotation.TableField;
import com.baomidou.mybatisplus.annotation.TableId;
import com.baomidou.mybatisplus.annotation.TableName;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.icetech.city.road.component.jackson.TemporalFormat;
import com.icetech.city.road.domain.entity.dto.CardTypeDto;
import com.icetech.city.road.utils.DateUtils;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.experimental.Accessors;

/**
 * <p>
 * 订单表
 * </p>
 *
 * @author wangzhiwei
 * @since 2021-03-04
 */
@Data
@EqualsAndHashCode(callSuper = false)
@Accessors(chain = true)
@TableName("ice_order")
public class Order implements Serializable {

    private static final long serialVersionUID = 1L;

    /**
     * 主键自增长id
     */
    @TableId(value = "id", type = IdType.AUTO)
    private Long id;

    /**
     * 道路id
     */
    private Integer parkId;

    /**
     * 订单号
     */
    private String orderNum;

    /**
     * 第三方记录ID
     */
    private String vendorRecordId;
    /**
     * 泊位id
     */
    private Integer berthId;

    /**
     * 车牌号
     */
    private String plateNum;

    /**
     * 车辆类型（1：临时车，默认1）
     */
    private Integer cardType;

    private Integer cardTypeId;

    /**
     * 进场时间
     */
    private Long enterTime;

    /**
     * 离场时间
     */
    private Long exitTime;

    /** 停车时长 */
    private Integer parkDuration;
    /**
     * 上报时间
     */
    private Long reportTime;

    /**
     * 车型（1：小型车，2：大型车，默认1）
     */
    private Integer carType;

    /**
     * 订单金额
     */
    private BigDecimal totalPrice;

    /**
     * 已缴金额
     */
    private BigDecimal paidPrice;

    /**
     * 待缴金额
     */
    private BigDecimal unpaidPrice;
    /**
     * 优惠金额
     */
    private BigDecimal discountPrice;
    /**
     * 退款金额
     */
    private BigDecimal refundPrice;
    /**
     * 退款标识(0-未退款;1-有退款)
     */
    private Integer refundFlag;

    /**
     * 订单类型（1：在场，2：离场）
     */
    private Integer orderType;

    /**
     * 订单状态（1：正常，2：已取消，3：异常，默认1）
     */
    private Integer orderStatus;

    /**
     * 异常或取消原因（orderStatus=2或3是必填)
     */
    private String errorCauses;

    /**
     * 缴费状态（1：未缴费2：预交费中，3：欠费，4：已缴费，5：无需缴费，6：未知）
     */
    private Integer payStatus;

    /**
     * 入场方式（（1：高位视频，2：地磁，3：pos机，4：人工手动，5：巡检车）
     */
    private Integer enterWay;

    /**
     * 入场设备序列号
     */
    private String enterDevice;

    /**
     * 出场方式（1：高位视频，2：地磁，3：pos机，4：人工手动，5：巡检车，6：自动离场）
     */
    private Integer exitWay;

    /**
     * 离场设备序列号
     */
    private String exitDevice;

    /**
     * 车牌颜色
     */
    private Integer plateColor;

    /**
     * 入场推送结果(0：未上传；9：上传成功；-9：上传失败)
     */
    @TableField("`push_enter_result`")
    private Integer pushEnterResult;
    /**
     * 出场推送结果(0：未上传；9：上传成功；-9：上传失败)
     */
    @TableField("`push_exit_result`")
    private Integer pushExitResult;

    /**
     * 添加时间
     */
    @TableField(updateStrategy = FieldStrategy.NEVER)
    private Date createTime;

    /**
     * 添加人
     */
    @TableField(updateStrategy = FieldStrategy.NEVER)
    private String adder;

    /**
     * 更新时间
     */
    private Date updateTime;

    /**
     * 更新人
     */
    private String updateUser;

    /**
     * 描述信息
     */
    private String remark;

    /**
     * 车场编号
     */
    @TableField(exist = false)
    private String parkCode;
    /**
     * 车场名称
     */
    @TableField(exist = false)
    private String parkName;

    /**
     * 开票状态（0:未开票；1:开票中；2:开票成功；-2:开票失败）
     */
    private Integer invoiceStatus;
    /** 最后打印用户 */
    private String lastPrintUser;
    /** 最后打印时间 */
    @JsonFormat(pattern = DateUtils.DEFAULT_DATETIME_PATTERN)
    @TemporalFormat(pattern = DateUtils.DEFAULT_DATETIME_PATTERN)
    private LocalDateTime lastPrintTime;

    /**
     * 泊位编号
     */
    @TableField(exist = false)
    private String berthCode;
    @TableField(exist = false)
    protected String enterVideo;
    @TableField(exist = false)
    protected String exitVideo;
    @TableField(exist = false)
    protected CardTypeDto cardTypeDto;

    public Order setDiscountPrice(BigDecimal discountPrice) {
        this.discountPrice = checkValue(discountPrice);
        return this;
    }

    public Order setUnpaidPrice(BigDecimal unpaidPrice) {
        this.unpaidPrice = checkValue(unpaidPrice);
        return this;
    }

    public Order setPaidPrice(BigDecimal paidPrice) {
        this.paidPrice = checkValue(paidPrice);
        return this;
    }

    public Order setTotalPrice(BigDecimal totalPrice) {
        this.totalPrice = checkValue(totalPrice);
        return this;
    }

    public Order setRefundPrice(BigDecimal refundPrice) {
        this.refundPrice = checkValue(refundPrice);
        return this;
    }

    private BigDecimal checkValue(BigDecimal value) {
        if (value == null || value.compareTo(BigDecimal.ZERO) < 0)
            return BigDecimal.ZERO;
        return value;
    }

    public Order setExitTimeWithDuration(Long exitTime) {
        return setExitTimeWithDuration(exitTime, this.enterTime);
    }

    public Order setExitTimeWithDuration(Long exitTime, Long enterTime) {
        setExitTime(exitTime);
        calcParkDuration(enterTime, exitTime);
        return this;
    }

    public Order calcParkDuration() {
        if (this.enterTime != null && this.exitTime != null) {
            calcParkDuration(this.enterTime, this.exitTime);
        }
        return this;
    }

    public Order calcParkDuration(Long enterTime, Long exitTime) {
        setParkDuration((int) (exitTime - enterTime));
        return this;
    }
}
