/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.client;

import java.io.IOException;
import java.util.Collection;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.atomic.AtomicInteger;
import org.eclipse.jetty.client.api.Connection;
import org.eclipse.jetty.client.api.Destination;
import org.eclipse.jetty.util.BlockingArrayQueue;
import org.eclipse.jetty.util.Promise;
import org.eclipse.jetty.util.component.ContainerLifeCycle;
import org.eclipse.jetty.util.component.Dumpable;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;

public class ConnectionPool
implements Dumpable {
    private static final Logger LOG = Log.getLogger(ConnectionPool.class);
    private final AtomicInteger connectionCount = new AtomicInteger();
    private final Destination destination;
    private final int maxConnections;
    private final Promise<Connection> connectionPromise;
    private final BlockingDeque<Connection> idleConnections;
    private final BlockingQueue<Connection> activeConnections;

    public ConnectionPool(Destination destination, int maxConnections, Promise<Connection> connectionPromise) {
        this.destination = destination;
        this.maxConnections = maxConnections;
        this.connectionPromise = connectionPromise;
        this.idleConnections = new LinkedBlockingDeque<Connection>(maxConnections);
        this.activeConnections = new BlockingArrayQueue(maxConnections);
    }

    public BlockingQueue<Connection> getIdleConnections() {
        return this.idleConnections;
    }

    public BlockingQueue<Connection> getActiveConnections() {
        return this.activeConnections;
    }

    public Connection acquire() {
        int next;
        int current;
        Connection result = this.acquireIdleConnection();
        if (result != null) {
            return result;
        }
        do {
            if ((next = (current = this.connectionCount.get()) + 1) <= this.maxConnections) continue;
            LOG.debug("Max connections {}/{} reached", new Object[]{current, this.maxConnections});
            return this.acquireIdleConnection();
        } while (!this.connectionCount.compareAndSet(current, next));
        LOG.debug("Connection {}/{} creation", new Object[]{next, this.maxConnections});
        this.destination.newConnection(new Promise<Connection>(){

            public void succeeded(Connection connection) {
                LOG.debug("Connection {}/{} creation succeeded {}", new Object[]{next, ConnectionPool.this.maxConnections, connection});
                ConnectionPool.this.activate(connection);
                ConnectionPool.this.connectionPromise.succeeded((Object)connection);
            }

            public void failed(Throwable x) {
                LOG.debug("Connection " + next + "/" + ConnectionPool.this.maxConnections + " creation failed", x);
                ConnectionPool.this.connectionCount.decrementAndGet();
                ConnectionPool.this.connectionPromise.failed(x);
            }
        });
        return this.acquireIdleConnection();
    }

    private Connection acquireIdleConnection() {
        Connection connection = (Connection)this.idleConnections.pollFirst();
        if (connection != null) {
            this.activate(connection);
        }
        return connection;
    }

    private boolean activate(Connection connection) {
        if (this.activeConnections.offer(connection)) {
            LOG.debug("Connection active {}", new Object[]{connection});
            return true;
        }
        LOG.debug("Connection active overflow {}", new Object[]{connection});
        return false;
    }

    public boolean release(Connection connection) {
        if (this.activeConnections.remove(connection)) {
            if (this.idleConnections.offerFirst(connection)) {
                LOG.debug("Connection idle {}", new Object[]{connection});
                return true;
            }
            LOG.debug("Connection idle overflow {}", new Object[]{connection});
        }
        return false;
    }

    public boolean remove(Connection connection) {
        boolean removed = this.activeConnections.remove(connection);
        if (removed |= this.idleConnections.remove(connection)) {
            int pooled = this.connectionCount.decrementAndGet();
            LOG.debug("Connection removed {} - pooled: {}", new Object[]{connection, pooled});
        }
        return removed;
    }

    public boolean isActive(Connection connection) {
        return this.activeConnections.contains(connection);
    }

    public boolean isIdle(Connection connection) {
        return this.idleConnections.contains(connection);
    }

    public void close() {
        for (Connection connection : this.idleConnections) {
            connection.close();
        }
        this.idleConnections.clear();
        for (Connection connection : this.activeConnections) {
            connection.close();
        }
        this.activeConnections.clear();
        this.connectionCount.set(0);
    }

    public String dump() {
        return ContainerLifeCycle.dump((Dumpable)this);
    }

    public void dump(Appendable out, String indent) throws IOException {
        ContainerLifeCycle.dumpObject((Appendable)out, (Object)this);
        ContainerLifeCycle.dump((Appendable)out, (String)indent, (Collection[])new Collection[]{this.activeConnections, this.idleConnections});
    }

    public String toString() {
        return String.format("%s %d/%d", this.getClass().getSimpleName(), this.connectionCount.get(), this.maxConnections);
    }
}

